;==========================================================================
;Version history:
;==========================================================================
;2010/01/14     First edition
;2010/07/26     Add the functions of speech event for SW_ChA/SW_ChB
;               Remove the interrupt mode of NMI on SW_ChA
;2010/12/01     Check if any single note/drum is played before turning off DAC in "F_ChA_DisInt" and "F_ChB_DisInt"
;2011/01/20     Rename "F_ChA_DisInt" as "F_ChA_PlayEnd" and "F_ChB_DisInt" as "F_ChB_PlayEnd"
;               Move "F_ChA_GetEvtAddr" and "F_ChB_GetEvtAddr" from libraries to SW_Channel.asm
;               Call the subroutine "F_TurnOnDAC" before playing a speech for controlling external amplifier
;               Revise "F_ChA_ServiceEvt" and "F_ChB_ServiceEvt"
;2011/02/23     Debug "F_ChA_GetEvtAddr" and "F_ChB_GetEvtAddr" to fix calculation error of event address
;2011/03/30     Add new function of IO event to "F_ChA_DealWithEvt" and "F_ChB_DealWithEvt" (controlled by "ChA_IOEvt" and "ChB_IOEvt")
;==========================================================================
        .INCLUDE        Inc\SW_Channel.inc

        .IF SW_ChA = ON
        .INCLUDE        Inc\SW_ChA.inc
        .ENDIF

        .IF SW_ChB = ON
        .INCLUDE        Inc\SW_ChB.inc
        .ENDIF
;==========================================================================
        .PUBLIC         F_TurnOnDAC
        .PUBLIC         T_SChSpeechFreqTable
;==========================================================================
        .CODE
;==========================================================================
T_SChSpeechFreqTable:
        %LoadSChSpeechFreq      8000    ;Index = 00H
        %LoadSChSpeechFreq      9000    ;Index = 01H
        %LoadSChSpeechFreq      10000   ;Index = 02H
        %LoadSChSpeechFreq      12000   ;Index = 03H
        %LoadSChSpeechFreq      14000   ;Index = 04H
        %LoadSChSpeechFreq      16000   ;Index = 05H
        %LoadSChSpeechFreq      18000   ;Index = 06H
        %LoadSChSpeechFreq      20000   ;Index = 07H
        %LoadSChSpeechFreq      22000   ;Index = 08H
        %LoadSChSpeechFreq      24000   ;Index = 09H
        %LoadSChSpeechFreq      26000   ;Index = 0AH
D_MaxSchFreq:   .EQU    ($-T_SChSpeechFreqTable)/2
;==========================================================================
;Purpose: Turn on the push-pull DAC
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_TurnOnDAC:
        LDA     P_DAC_Ctrl
        AND     #(D_DAC_Enable+D_OPN_En+D_OPP_En)
        CMP     #(D_DAC_Enable+D_OPN_En+D_OPP_En)
        BEQ     L_Next?
        %TurnOnDAC
L_Next?:
        RTS
;==========================================================================
;Purpose: Used for enabling interrupt before playing a speech
;Input: None
;Return: None
;Destroy: A
;==========================================================================
        .IF SW_ChA = ON
F_ChA_EnInt:                            ;Before playing a speech, the library will call this function to start interrupt
        .IF ChA_Evt = ON && ChA_IOEvt = ON
        JSR     F_IOEventEnd            ;Disable all software PWMIOs
        .ENDIF

        LDA     P_INT_CtrlL
        ORA     #D_TMAIntEn
        STA     P_INT_CtrlL
        CLI
        RTS
;==========================================================================
;Purpose: Disable interrupt after a speech is completed and turn off DAC if no sound is played
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_ChA_PlayEnd:                          ;After a speech is completed, the library will call this function to stop interrupt
        LDA     P_TMA_Ctrl
        AND     #~D_TMR_Enable
        STA     P_TMA_Ctrl

        LDA     P_INT_CtrlL
        AND     #~D_TMAIntEn
        STA     P_INT_CtrlL

        .IF ChA_Evt = ON && ChA_IOEvt = ON
        JSR     F_IOEventEnd            ;Disable all software PWMIOs
        .ENDIF

        .IF SPEECH = ON
        LDA     R_SpeechStatus
        BNE     L_ChA_Exit?
        .ENDIF

        .IF MIDI = ON
        LDA     R_MIDIStatus            ;Check if MIDI is playing
        AND     #D_MIDIEn
        BNE     L_ChA_Exit?

        LDA     R_SingleNote            ;Check if any single note is played
        ORA     R_SingleDrum            ;Check if any single drum is played
        BNE     L_ChA_Exit?
        .ENDIF

        .IF SW_ChB = ON
        %IsActiveChB                    ;Check if SW_ChB is active
        BCS     L_ChA_Exit?
        .ENDIF

        .IF ComAir = ON
        JSR     F_CA_IsActive
        BCS     L_ChA_Exit?
        .ENDIF

        %TurnOffDAC                     ;Turn off DAC
L_ChA_Exit?:
        RTS
;==========================================================================
;Purpose: Get the address of event table
;Input: Speech address (A = AddrL, X = AddrH, Y = AddrB)
;Return: None
;Destroy: A, Y
;==========================================================================
F_ChA_GetEvtAddr:
            .IF ChA_Evt = ON
        STA     R_ChA_EvtAddrL          ;Speech address low byte
        STX     R_ChA_EvtAddrH          ;Speech address high byte
        CPX     #C0H
        BCS     L_NotCommonBank?
        LDA     #00H                    ;If the high byte belongs to common bank, get the bank number
        STA     R_ChA_EvtAddrB
        TXA
        ASL     A
        ROL     R_ChA_EvtAddrB
        ASL     A
        ROL     R_ChA_EvtAddrB
        LDY     R_ChA_EvtAddrB
        JMP     L_GetOffset?
L_NotCommonBank?:
        STY     R_ChA_EvtAddrB          ;Speech address bank number
L_GetOffset?:
        STY     P_Bank
        LDY     #03H
        LDA     (R_ChA_EvtAddrL),Y      ;Get offset high byte of event address
        STA     R_ChA_EvtTag
        DEY
        LDA     (R_ChA_EvtAddrL),Y      ;Get offset middle byte of event address
        STA     R_ChA_MainIndex
        DEY
        LDA     (R_ChA_EvtAddrL),Y      ;Get offset low byte of event address
        STA     R_ChA_SubIndex

        LDA     R_ChA_EvtAddrH          ;Transfrom speech address from CPU view to ROM view
        AND     #00111111B              ;Clear 2-bit MSB of high byte
        STA     R_ChA_EvtAddrH
        LSR     R_ChA_EvtAddrB          ;Get 2-bit LSB of bank number
        ROR     A
        LSR     R_ChA_EvtAddrB
        ROR     A
        AND     #11000000B
        ORA     R_ChA_EvtAddrH          ;Speech high byte = R_ChA_EvtAddrB[1:0]:R_ChA_EvtAddrH[5:0]
        STA     R_ChA_EvtAddrH

        CLC                             ;Speech address + offset
        LDA     R_ChA_EvtAddrL
        ADC     R_ChA_SubIndex
        STA     R_ChA_EvtAddrL
        LDA     R_ChA_EvtAddrH
        ADC     R_ChA_MainIndex
        STA     R_ChA_EvtAddrH
        LDA     R_ChA_EvtAddrB
        ADC     R_ChA_EvtTag
        STA     R_ChA_EvtAddrB

        LDA     R_ChA_EvtAddrH          ;Transfrom speech address from ROM view to CPU view
        ASL     A
        ROL     R_ChA_EvtAddrB
        ASL     A
        ROL     R_ChA_EvtAddrB
        LDA     R_ChA_EvtAddrB
        CMP     #03H
        BCC     L_SetBank?
        LDA     R_ChA_EvtAddrH          ;Set address to bank window
        ORA     #11000000B
        STA     R_ChA_EvtAddrH
        JMP     L_Exit?
L_SetBank?:
        LDA     #03H
        STA     R_ChA_EvtAddrB
L_Exit?:
            .ENDIF
        RTS
;==========================================================================
;Purpose: Service for speech event on SW_ChA (Put into the main loop)
;Input: None
;Return: None
;Destroy: A, X, Y
;==========================================================================
            .IF ChA_Evt = ON
F_ChA_ServiceEvt:
        %IsActiveChA
        BCC     L_Exit?
        %IsEventChA                     ;Check if any event is detected
        BCC     L_Exit?
        %ClrEvtFlagChA                  ;Clear event flag to allow a new event issuing (A = Event tag)
        JSR     F_ChA_GetEvt
L_Exit?:
        RTS
;==========================================================================
F_ChA_GetEvt:
        STA     R_ChA_EvtTag            ;Backup event tag
L_ChA_EvtLoop?:
        LDY     #00H
        LDA     R_ChA_EvtAddrB
        SEI
        STA     P_Bank
        LDA     (R_ChA_EvtAddrL),Y      ;Get event frame header
        CLI
        TAX
        AND     #00000011B
        CMP     R_ChA_EvtTag            ;Check if the two event tags are equal
        BEQ     L_ChA_GetEvt?
        JSR     F_ChA_ProcEvt           ;If the two tags are not equal, proecss old events first
        JMP     L_ChA_EvtLoop?
L_ChA_GetEvt?:
        JSR     F_ChA_ProcEvt           ;If the two tags are equal, process current event
        RTS
;==========================================================================
F_ChA_ProcEvt:
        INC     R_ChA_EvtAddrL          ;Address + 1
        BNE     L_NoCarry?
        INC     R_ChA_EvtAddrH
        BNE     L_NoCarry?
        INC     R_ChA_EvtAddrB
        LDA     #C0H
        STA     R_ChA_EvtAddrH
L_NoCarry?:
        TXA                             ;Get event number
        LSR     A
        LSR     A
        STA     R_ChA_EvtNumber
L_GetEvtLoop?:
        LDY     #00H
        LDA     R_ChA_EvtAddrB
        SEI
        STA     P_Bank
        LDA     (R_ChA_EvtAddrL),Y      ;Get main index of an event
        CLI
        STA     R_ChA_MainIndex
        INC     R_ChA_EvtAddrL          ;Address + 1
        BNE     L_GetNext?
        INC     R_ChA_EvtAddrH
        BNE     L_GetNext?
        INC     R_ChA_EvtAddrB
        LDA     #C0H
        STA     R_ChA_EvtAddrH
L_GetNext?:
        LDA     R_ChA_EvtAddrB
        SEI
        STA     P_Bank
        LDA     (R_ChA_EvtAddrL),Y      ;Get sub index of an event
        CLI
        STA     R_ChA_SubIndex
        INC     R_ChA_EvtAddrL          ;Address + 1
        BNE     L_Next?
        INC     R_ChA_EvtAddrH
        BNE     L_Next?
        INC     R_ChA_EvtAddrB
        LDA     #C0H
        STA     R_ChA_EvtAddrH
L_Next?:
        JSR     F_ChA_DealWithEvt       ;Deal with an event
        DEC     R_ChA_EvtNumber
        BNE     L_GetEvtLoop?
        RTS
;==========================================================================
;Purpose: Deal with a speech event on SW_ChA
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_ChA_DealWithEvt:
                .IF ChA_IOEvt = ON
        LDA     R_ChA_MainIndex
        CMP     #80H
        BCC     L_ChA_UserEvent?
        CMP     #A0H
        BCC     L_ChA_IOEvent?
        CMP     #F0H
        BEQ     L_ChA_IOEventStart?
        CMP     #F1H
        BEQ     L_ChA_IOEventEnd?
        RTS

L_ChA_UserEvent?:
        LDA     R_ChA_MainIndex         ;Process user event
        NOP
        LDA     R_ChA_SubIndex
        NOP
        RTS
L_ChA_IOEvent?:
        LDY     R_ChA_SubIndex
        JSR     F_IOEventProcess        ;Start a PWMIO
        RTS
L_ChA_IOEventStart?:
        JSR     F_IOEventEnd            ;Set all PWMIOs as output low
        RTS
L_ChA_IOEventEnd?:
        JSR     F_IOEventEnd            ;Set all PWMIOs as output low
        RTS
                .ELSE
        LDA     R_ChA_MainIndex
        LDA     R_ChA_SubIndex
        RTS
                .ENDIF
            .ENDIF
        .ENDIF
;==========================================================================
;Purpose: Used for enabling interrupt before playing a speech
;Input: None
;Return: None
;Destroy: A
;==========================================================================
        .IF SW_ChB = ON
F_ChB_EnInt:                            ;Before playing a speech, the library will call this function to start interrupt
        .IF ChB_Evt = ON && ChB_IOEvt = ON
        JSR     F_IOEventEnd            ;Disable all software PWMIOs
        .ENDIF

        LDA     P_INT_CtrlL
        ORA     #D_TMBIntEn
        STA     P_INT_CtrlL
        CLI
        RTS
;==========================================================================
;Purpose: Disable interrupt after a speech is completed and turn off DAC if no sound is played
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_ChB_PlayEnd:                          ;After a speech is completed, the library will call this function to stop interrupt
        LDA     P_TMB_Ctrl
        AND     #~D_TMR_Enable
        STA     P_TMB_Ctrl

        LDA     P_INT_CtrlL
        AND     #~D_TMBIntEn
        STA     P_INT_CtrlL

        .IF ChB_Evt = ON && ChB_IOEvt = ON
        JSR     F_IOEventEnd            ;Disable all software PWMIOs
        .ENDIF

        .IF SPEECH = ON
        LDA     R_SpeechStatus
        BNE     L_ChB_Exit?
        .ENDIF

        .IF MIDI = ON
        LDA     R_MIDIStatus            ;Check if MIDI is playing
        AND     #D_MIDIEn
        BNE     L_ChB_Exit?

        LDA     R_SingleNote            ;Check if any single note is played
        ORA     R_SingleDrum            ;Check if any single drum is played
        BNE     L_ChB_Exit?
        .ENDIF

        .IF SW_ChA = ON
        %IsActiveChA                    ;Check if SW_ChA is active
        BCS     L_ChB_Exit?
        .ENDIF

        .IF ComAir = ON
        JSR     F_CA_IsActive
        BCS     L_ChB_Exit?
        .ENDIF

        %TurnOffDAC                     ;Turn off DAC
L_ChB_Exit?:
        RTS
;==========================================================================
;Purpose: Get the address of event table
;Input: Speech address (A = AddrL, X = AddrH, Y = AddrB)
;Return: None
;Destroy: A, Y
;==========================================================================
F_ChB_GetEvtAddr:
            .IF ChB_Evt = ON
        STA     R_ChB_EvtAddrL          ;Speech address low byte
        STX     R_ChB_EvtAddrH          ;Speech address high byte
        CPX     #C0H
        BCS     L_NotCommonBank?
        LDA     #00H                    ;If the high byte belongs to common bank, get the bank number
        STA     R_ChB_EvtAddrB
        TXA
        ASL     A
        ROL     R_ChB_EvtAddrB
        ASL     A
        ROL     R_ChB_EvtAddrB
        LDY     R_ChB_EvtAddrB
        JMP     L_GetOffset?
L_NotCommonBank?:
        STY     R_ChB_EvtAddrB          ;Speech address bank number
L_GetOffset?:
        STY     P_Bank
        LDY     #03H
        LDA     (R_ChB_EvtAddrL),Y      ;Get offset high byte of event address
        STA     R_ChB_EvtTag
        DEY
        LDA     (R_ChB_EvtAddrL),Y      ;Get offset middle byte of event address
        STA     R_ChB_MainIndex
        DEY
        LDA     (R_ChB_EvtAddrL),Y      ;Get offset low byte of event address
        STA     R_ChB_SubIndex

        LDA     R_ChB_EvtAddrH          ;Transfrom speech address from CPU view to ROM view
        AND     #00111111B              ;Clear 2-bit MSB of high byte
        STA     R_ChB_EvtAddrH
        LSR     R_ChB_EvtAddrB          ;Get 2-bit LSB of bank number
        ROR     A
        LSR     R_ChB_EvtAddrB
        ROR     A
        AND     #11000000B
        ORA     R_ChB_EvtAddrH          ;Speech high byte = R_ChB_EvtAddrB[1:0]:R_ChB_EvtAddrH[5:0]
        STA     R_ChB_EvtAddrH

        CLC                             ;Speech address + offset
        LDA     R_ChB_EvtAddrL
        ADC     R_ChB_SubIndex
        STA     R_ChB_EvtAddrL
        LDA     R_ChB_EvtAddrH
        ADC     R_ChB_MainIndex
        STA     R_ChB_EvtAddrH
        LDA     R_ChB_EvtAddrB
        ADC     R_ChB_EvtTag
        STA     R_ChB_EvtAddrB

        LDA     R_ChB_EvtAddrH          ;Transfrom speech address from ROM view to CPU view
        ASL     A
        ROL     R_ChB_EvtAddrB
        ASL     A
        ROL     R_ChB_EvtAddrB
        LDA     R_ChB_EvtAddrB
        CMP     #03H
        BCC     L_SetBank?
        LDA     R_ChB_EvtAddrH          ;Set address to bank window
        ORA     #11000000B
        STA     R_ChB_EvtAddrH
        JMP     L_Exit?
L_SetBank?:
        LDA     #03H
        STA     R_ChB_EvtAddrB
L_Exit?:
            .ENDIF
        RTS
;==========================================================================
;Purpose: Service for speech event on SW_ChB (Put into the main loop)
;Input: None
;Return: None
;Destroy: A, X, Y
;==========================================================================
            .IF ChB_Evt = ON
F_ChB_ServiceEvt:
        %IsActiveChB
        BCC     L_Exit?
        %IsEventChB                     ;Check if any event is detected
        BCC     L_Exit?
        %ClrEvtFlagChB                  ;Clear event flag to allow a new event issuing (A = Event tag)
        JSR     F_ChB_GetEvt
L_Exit?:
        RTS
;==========================================================================
F_ChB_GetEvt:
        STA     R_ChB_EvtTag            ;Backup event tag
L_ChB_EvtLoop?:
        LDY     #00H
        LDA     R_ChB_EvtAddrB
        SEI
        STA     P_Bank
        LDA     (R_ChB_EvtAddrL),Y      ;Get event frame header
        CLI
        TAX
        AND     #00000011B
        CMP     R_ChB_EvtTag            ;Check if the two event tags are equal
        BEQ     L_ChB_GetEvt?
        JSR     F_ChB_ProcEvt           ;If the two tags are not equal, proecss old events first
        JMP     L_ChB_EvtLoop?
L_ChB_GetEvt?:
        JSR     F_ChB_ProcEvt           ;If the two tags are equal, process current event
        RTS
;==========================================================================
F_ChB_ProcEvt:
        INC     R_ChB_EvtAddrL          ;Address + 1
        BNE     L_NoCarry?
        INC     R_ChB_EvtAddrH
        BNE     L_NoCarry?
        INC     R_ChB_EvtAddrB
        LDA     #C0H
        STA     R_ChB_EvtAddrH
L_NoCarry?:
        TXA                             ;Get event number
        LSR     A
        LSR     A
        STA     R_ChB_EvtNumber
L_GetEvtLoop?:
        LDY     #00H
        LDA     R_ChB_EvtAddrB
        SEI
        STA     P_Bank
        LDA     (R_ChB_EvtAddrL),Y      ;Get main index of an event
        CLI
        STA     R_ChB_MainIndex
        INC     R_ChB_EvtAddrL          ;Address + 1
        BNE     L_GetNext?
        INC     R_ChB_EvtAddrH
        BNE     L_GetNext?
        INC     R_ChB_EvtAddrB
        LDA     #C0H
        STA     R_ChB_EvtAddrH
L_GetNext?:
        LDA     R_ChB_EvtAddrB
        SEI
        STA     P_Bank
        LDA     (R_ChB_EvtAddrL),Y      ;Get sub index of an event
        CLI
        STA     R_ChB_SubIndex
        INC     R_ChB_EvtAddrL          ;Address + 1
        BNE     L_Next?
        INC     R_ChB_EvtAddrH
        BNE     L_Next?
        INC     R_ChB_EvtAddrB
        LDA     #C0H
        STA     R_ChB_EvtAddrH
L_Next?:
        JSR     F_ChB_DealWithEvt       ;Deal with an event
        DEC     R_ChB_EvtNumber
        BNE     L_GetEvtLoop?
        RTS
;==========================================================================
;Purpose: Deal with a speech event on SW_ChB
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_ChB_DealWithEvt:
                .IF ChB_IOEvt = ON
        LDA     R_ChB_MainIndex
        CMP     #80H
        BCC     L_ChB_UserEvent?
        CMP     #A0H
        BCC     L_ChB_IOEvent?
        CMP     #F0H
        BEQ     L_ChB_IOEventStart?
        CMP     #F1H
        BEQ     L_ChB_IOEventEnd?
        RTS

L_ChB_UserEvent?:
        LDA     R_ChB_MainIndex         ;Process user event
        NOP
        LDA     R_ChB_SubIndex
        NOP
        RTS
L_ChB_IOEvent?:
        LDY     R_ChB_SubIndex
        JSR     F_IOEventProcess        ;Start a PWMIO
        RTS
L_ChB_IOEventStart?:
        JSR     F_IOEventEnd            ;Set all PWMIOs as output low
        RTS
L_ChB_IOEventEnd?:
        JSR     F_IOEventEnd            ;Set all PWMIOs as output low
        RTS
                .ELSE
        LDA     R_ChB_MainIndex
        LDA     R_ChB_SubIndex
        RTS
                .ENDIF
            .ENDIF
        .ENDIF
;==========================================================================
